//=============================================================================
// MPP_ChangeBattleMessageSpeed.js
//=============================================================================
// Copyright (c) 2024 Mokusei Penguin
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MZ
 * @plugindesc Changes the speed of battle messages.
 * @author Mokusei Penguin
 * @url 
 * 
 * @orderAfter MPP_SmoothBattleLog
 *
 * @help [version 1.0.0]
 * - This plugin is for RPG Maker MZ.
 * 
 * ▼ Plugin command details
 *  - In the item to enter a numerical value, select the text and
 *    write v[N] to refer to the variable N.
 * 
 * ================================
 * Mail : wood_penguin＠yahoo.co.jp (＠ is half-width)
 * Blog : http://woodpenguin.blog.fc2.com/
 * License : MIT license
 *
 *  @command setMessageSpeed
 *      @desc 
 *      @arg speed
 *          @desc 
 *          @type number
 *              @min 0
 *              @max 999
 *          @default 16
 * 
 *  @param Default Message Speed
 *      @desc 
 *      @type number
 *          @min 0
 *          @max 999
 *      @default 16
 * 
 */

/*:ja
 * @target MZ
 * @plugindesc 戦闘メッセージの速度を変更します。
 * @author 木星ペンギン
 * @url 
 * 
 * @orderAfter MPP_SmoothBattleLog
 *
 * @help [version 1.0.0]
 * - このプラグインはRPGツクールMZ用です。
 * 
 * ▼ プラグインコマンド
 *  - MVでは数値を入力する項目で v[N] と記述することで変数N番を参照します。
 * 
 * ================================
 * Mail : wood_penguin＠yahoo.co.jp (＠は半角)
 * Blog : http://woodpenguin.blog.fc2.com/
 * License : MIT license
 *
 *  @command setMessageSpeed
 *      @text メッセージ速度設定
 *      @desc 
 *      @arg speed
 *          @text 速度
 *          @desc 
 *          @type number
 *              @min 0
 *              @max 999
 *          @default 16
 * 
 *  @param Default Message Speed
 *      @text メッセージ速度初期値
 *      @desc 
 *      @type number
 *          @min 0
 *          @max 999
 *      @default 16
 * 
 */

(() => {
    'use strict';

    const pluginName = 'MPP_ChangeBattleMessageSpeed';
    
    // Plugin Parameters
    const parameters = PluginManager.parameters(pluginName);
    const param_DefaultMessageSpeed = Number(parameters['Default Message Speed'] || 0);

    //-------------------------------------------------------------------------
    // Window_BattleLog

    Window_BattleLog.prototype.messageSpeed = function() {
        if ('battleMessageSpeed' in $gameSystem) {
            return $gameSystem.battleMessageSpeed;
        }
        return param_DefaultMessageSpeed;
    };

    //-------------------------------------------------------------------------
    // PluginManager

    PluginManager.registerCommand(pluginName, 'setMessageSpeed', args => {
        const speed = PluginManager.mppValue(args.speed);
        $gameSystem.battleMessageSpeed = speed;
    });

    PluginManager.mppValue = function(value) {
        const match = /^V\[(\d+)\]$/i.exec(value);
        return match ? $gameVariables.value(+match[1]) : +value;
    };

})();
